/*
 * Decompiled with CFR 0.152.
 */
package de.gfz_potsdam.gipp.common.seis.miniseed;

import de.gfz_potsdam.gipp.common.io.IOUtils;
import de.gfz_potsdam.gipp.common.seis.miniseed.EncodingFormat;
import de.gfz_potsdam.gipp.common.seis.miniseed.IeeeDoubleCodec;
import de.gfz_potsdam.gipp.common.seis.miniseed.IeeeFloatCodec;
import de.gfz_potsdam.gipp.common.seis.miniseed.IntegerCodec;
import de.gfz_potsdam.gipp.common.seis.miniseed.IntegrityException;
import de.gfz_potsdam.gipp.common.seis.miniseed.MiniseedCodec;
import de.gfz_potsdam.gipp.common.seis.miniseed.SteimOneCodec;
import de.gfz_potsdam.gipp.common.seis.miniseed.SteimTwoCodec;
import de.gfz_potsdam.gipp.common.seis.miniseed.UnsupportedException;
import de.gfz_potsdam.gipp.common.string.StringUtils;
import de.gfz_potsdam.gipp.common.time.TimeMoment;
import de.gfz_potsdam.gipp.common.time.TimeSpan;
import java.io.EOFException;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.charset.StandardCharsets;
import java.security.MessageDigest;
import java.util.Arrays;
import java.util.Collection;
import java.util.Iterator;
import java.util.SortedSet;

public class Record
implements Comparable<Record> {
    private static final int MINIMUM_RECORD_SIZE = 64;
    public static final int DEFAULT_RECORD_SIZE = 4096;
    private static final int MICROSECONDS_PER_TICK = 100;
    private static final int NANOSECONDS_PER_TICK = 100000;
    protected ByteBuffer buffer;
    private int beginBlockette100;
    private int beginBlockette1000;
    private int beginBlockette1001;

    public Record() {
        this(4096, ByteOrder.nativeOrder(), EncodingFormat.STEIM_1, false);
    }

    public Record(int recordSize, ByteOrder byteOrder, EncodingFormat dataCodec, boolean useBlockette1001) throws IllegalArgumentException {
        byte exponent = (byte)Math.round(Math.ceil(Math.log(recordSize) / Math.log(2.0)));
        if (exponent < 6) {
            throw new IllegalArgumentException("A miniSEED record must be at least 64 bytes long (only " + recordSize + "bytes were requested).");
        }
        if (exponent > 31) {
            throw new IllegalArgumentException("A miniSEED record may not be larger than 2147483646 bytes ( " + recordSize + " bytes were requested).");
        }
        this.buffer = ByteBuffer.allocate(1 << exponent);
        this.beginBlockette1000 = FixedHeader.totalSize();
        this.beginBlockette1001 = useBlockette1001 ? this.beginBlockette1000 + Blockette1000.totalSize() : -1;
        this.setByteOrder(byteOrder);
        if (!useBlockette1001) {
            this.buffer.put(FixedHeader.DATA_INDICATOR.offset(), (byte)68);
            this.buffer.put(FixedHeader.SPACE.offset(), (byte)32);
            this.buffer.putShort(FixedHeader.BEGIN_DATA.offset(), (short)64);
            this.buffer.put(FixedHeader.BLOCKETTES.offset(), (byte)1);
            this.buffer.putShort(FixedHeader.BEGIN_BLOCKETTE.offset(), (short)this.findBlockette(1000, this.buffer.capacity()));
            this.buffer.putShort(this.findBlockette(1000, this.buffer.capacity()) + Blockette1000.BLOCKETTE_ID.offset(), (short)1000);
            this.buffer.putShort(this.findBlockette(1000, this.buffer.capacity()) + Blockette1000.NEXT_BLOCKETTE.offset(), (short)0);
            this.buffer.put(this.findBlockette(1000, this.buffer.capacity()) + Blockette1000.DATA_FORMAT.offset(), dataCodec.getCode());
            this.buffer.put(this.findBlockette(1000, this.buffer.capacity()) + Blockette1000.RECORD_LENGTH.offset(), exponent);
        } else {
            this.buffer.put(FixedHeader.DATA_INDICATOR.offset(), (byte)68);
            this.buffer.put(FixedHeader.SPACE.offset(), (byte)32);
            this.buffer.putShort(FixedHeader.BEGIN_DATA.offset(), (short)64);
            this.buffer.put(FixedHeader.BLOCKETTES.offset(), (byte)2);
            this.buffer.putShort(FixedHeader.BEGIN_BLOCKETTE.offset(), (short)this.findBlockette(1000, this.buffer.capacity()));
            this.buffer.putShort(this.findBlockette(1000, this.buffer.capacity()) + Blockette1000.BLOCKETTE_ID.offset(), (short)1000);
            this.buffer.putShort(this.findBlockette(1000, this.buffer.capacity()) + Blockette1000.NEXT_BLOCKETTE.offset(), (short)this.findBlockette(1001, this.buffer.capacity()));
            this.buffer.put(this.findBlockette(1000, this.buffer.capacity()) + Blockette1000.DATA_FORMAT.offset(), dataCodec.getCode());
            this.buffer.put(this.findBlockette(1000, this.buffer.capacity()) + Blockette1000.RECORD_LENGTH.offset(), exponent);
            this.buffer.putShort(this.findBlockette(1001, this.buffer.capacity()) + Blockette1001.BLOCKETTE_ID.offset(), (short)1001);
            this.buffer.putShort(this.findBlockette(1001, this.buffer.capacity()) + Blockette1001.NEXT_BLOCKETTE.offset(), (short)0);
            this.buffer.put(this.findBlockette(1001, this.buffer.capacity()) + Blockette1001.TIMING_QUALITY.offset(), (byte)0);
            this.buffer.put(this.findBlockette(1001, this.buffer.capacity()) + Blockette1001.TIME_MICROSECOND.offset(), (byte)0);
            this.buffer.put(this.findBlockette(1001, this.buffer.capacity()) + Blockette1001.DATA_FRAMES.offset(), (byte)0);
        }
    }

    public Record(byte[] array) {
        if (array.length < 64) {
            throw new IntegrityException("MiniSEED record is to short! An empty, \"no data\" record is at least 64 Bytes long.");
        }
        this.beginBlockette100 = 0;
        this.beginBlockette1000 = 0;
        this.beginBlockette1001 = 0;
        this.buffer = ByteBuffer.wrap(array);
        if (!this.beginsWithMagicNumber()) {
            throw new IntegrityException("MiniSEED record does not begin with a six digit number followed by the character 'D', 'R', 'Q' or 'M' and a space character (e.g. '123456D').");
        }
        this.buffer.order(this.getByteOrder());
    }

    private static String sanitizeText(String dirty) {
        assert (dirty != null) : "Cannot sanitize a 'null' string!";
        char[] result = new char[dirty.length()];
        int count = 0;
        for (int n : dirty.toCharArray()) {
            if (Character.isISOControl((char)n) || n == File.separatorChar) continue;
            result[count++] = Character.isWhitespace((char)n) ? 32 : n;
        }
        if (count == dirty.length()) {
            return dirty.trim();
        }
        return new String(result, 0, count).trim();
    }

    private void pokeText(int offset, int length, String text) throws IllegalArgumentException {
        assert (length >= 0) : "Negative text length!";
        if (text == null) {
            throw new IllegalArgumentException("No text given (got 'null').");
        }
        if (text.length() > length) {
            throw new IllegalArgumentException("Text string \"" + text + "\" is longer than the allowed " + length + " bytes.");
        }
        byte[] array = text.getBytes(StandardCharsets.US_ASCII);
        int blank = 32;
        for (int i = 0; i < length; ++i) {
            if (i < text.length()) {
                this.buffer.put(offset + i, array[i]);
                continue;
            }
            this.buffer.put(offset + i, (byte)32);
        }
    }

    private String appendBitMaskInfo(StringBuilder textLine, int flag, int bitMask, String bitDescription) {
        assert (bitDescription != null) : "Calling this method with a 'null' description makes no sense!";
        StringBuilder builder = textLine == null ? new StringBuilder() : textLine;
        if ((flag & bitMask) == bitMask) {
            if (builder.length() > 0) {
                builder.append(", ");
            }
            builder.append(bitDescription);
        }
        return builder.toString();
    }

    public static boolean isSameStream(Record ... records) {
        if (records.length <= 0) {
            return false;
        }
        if (records[0] == null) {
            return false;
        }
        if (records.length == 1) {
            return true;
        }
        for (int i = 1; i < records.length; ++i) {
            if (records[i] == null) {
                return false;
            }
            if (records[i].getNetworkId().equals(records[i - 1].getNetworkId()) && records[i].getLocationId().equals(records[i - 1].getLocationId()) && records[i].getStationId().equals(records[i - 1].getStationId()) && records[i].getChannelId().equals(records[i - 1].getChannelId()) && records[i].getSampleRate() == records[i - 1].getSampleRate()) continue;
            return false;
        }
        return true;
    }

    public static boolean isSameStream(Collection<Record> records) {
        if (records.isEmpty()) {
            return false;
        }
        if (records.size() == 1) {
            return true;
        }
        Iterator<Record> iter = records.iterator();
        Record reference = iter.next();
        while (iter.hasNext()) {
            Record current = iter.next();
            if (reference.getNetworkId().equals(current.getNetworkId()) && reference.getLocationId().equals(current.getLocationId()) && reference.getStationId().equals(current.getStationId()) && reference.getChannelId().equals(current.getChannelId()) && reference.getSampleRate() == current.getSampleRate()) continue;
            return false;
        }
        return true;
    }

    public static boolean isContinuous(Record ... records) {
        if (records.length <= 0) {
            return false;
        }
        if (records[0] == null) {
            return false;
        }
        if (records.length == 1) {
            return true;
        }
        TimeSpan epsilon = new TimeSpan.Builder().addSeconds(records[0].getSamplePeriod() / 4.0).build();
        return Record.isContinuous(epsilon, records);
    }

    public static boolean isContinuous(TimeSpan epsilon, Record ... records) {
        if (records.length <= 0) {
            return false;
        }
        if (records[0] == null) {
            return false;
        }
        if (records.length == 1) {
            return true;
        }
        if (!Record.isSameStream(records)) {
            return false;
        }
        for (int i = 1; i < records.length; ++i) {
            TimeMoment real;
            if (records[i] == null) {
                return false;
            }
            TimeMoment expected = records[i - 1].getTimeAt(records[i - 1].getSampleCount());
            if (TimeSpan.diff(expected, real = records[i].getStartTime()).compareTo(epsilon) <= 0) continue;
            return false;
        }
        return true;
    }

    public static boolean isContinuous(SortedSet<Record> records) {
        if (records.isEmpty()) {
            return false;
        }
        if (records.size() == 1) {
            return true;
        }
        TimeSpan epsilon = new TimeSpan.Builder().addSeconds(records.first().getSamplePeriod() / 4.0).build();
        return Record.isContinuous(epsilon, records);
    }

    public static boolean isContinuous(TimeSpan epsilon, SortedSet<Record> records) {
        if (records.isEmpty()) {
            return false;
        }
        if (records.size() == 1) {
            return true;
        }
        if (!Record.isSameStream(records)) {
            return false;
        }
        Iterator iter = records.iterator();
        Record previous = (Record)iter.next();
        while (iter.hasNext()) {
            Record current = (Record)iter.next();
            TimeMoment expected = previous.getTimeAt(previous.getSampleCount());
            TimeMoment real = current.getStartTime();
            previous = current;
            if (TimeSpan.diff(expected, real).compareTo(epsilon) <= 0) continue;
            return false;
        }
        return true;
    }

    public static Record duplicate(Record source) {
        assert (source.size() > 64) : "MiniSEED record to small!";
        byte[] buf = new byte[source.size()];
        System.arraycopy(source.buffer.array(), 0, buf, 0, source.size());
        return new Record(buf);
    }

    private int findBlockette(int id, int limit) throws IndexOutOfBoundsException {
        assert (limit <= this.buffer.capacity()) : "Search limit may not exceed buffer capacity!";
        if (id == 1000 && this.beginBlockette1000 != 0) {
            return this.beginBlockette1000;
        }
        if (id == 1001 && this.beginBlockette1001 != 0) {
            return this.beginBlockette1001;
        }
        if (id == 100 && this.beginBlockette100 != 0) {
            return this.beginBlockette100;
        }
        short blocketteStart = this.buffer.getShort(FixedHeader.BEGIN_BLOCKETTE.offset());
        while (blocketteStart != 0) {
            short blocketteId;
            if (blocketteStart < FixedHeader.totalSize() || blocketteStart > limit) {
                if ((blocketteStart = Short.reverseBytes(blocketteStart)) < FixedHeader.totalSize() || blocketteStart > limit) {
                    throw new IndexOutOfBoundsException("The blockette chain extends beyond the " + limit + " bytes limit given as argument.");
                }
                blocketteId = Short.reverseBytes(this.buffer.getShort(blocketteStart + DataBlockette.BLOCKETTE_ID.offset()));
            } else {
                blocketteId = this.buffer.getShort(blocketteStart + DataBlockette.BLOCKETTE_ID.offset());
            }
            switch (blocketteId) {
                case 100: {
                    this.beginBlockette100 = blocketteStart;
                    if (this.beginBlockette100 + Blockette100.totalSize() <= limit) break;
                    throw new IndexOutOfBoundsException("Blockette 100 partly extends beyond the " + limit + " bytes limit given as argument.");
                }
                case 1000: {
                    this.beginBlockette1000 = blocketteStart;
                    if (this.beginBlockette1000 + Blockette1000.totalSize() <= limit) break;
                    throw new IndexOutOfBoundsException("Blockette 1000 partly extends beyond the " + limit + " bytes limit given as argument.");
                }
                case 1001: {
                    this.beginBlockette1001 = blocketteStart;
                    if (this.beginBlockette1001 + Blockette1001.totalSize() <= limit) break;
                    throw new IndexOutOfBoundsException("Blockette 1001 partly extends beyond the " + limit + " bytes limit given as argument.");
                }
            }
            if (id == blocketteId) {
                return blocketteStart;
            }
            blocketteStart = this.buffer.getShort(blocketteStart + DataBlockette.NEXT_BLOCKETTE.offset());
        }
        if (id == 1000) {
            this.beginBlockette1000 = -1;
        } else if (id == 1001) {
            this.beginBlockette1001 = -1;
        } else if (id == 100) {
            this.beginBlockette100 = -1;
        }
        return -1;
    }

    private boolean injectBlockette1001() {
        short lastBlocketteSize;
        short nextBlocketteStart = this.buffer.getShort(FixedHeader.BEGIN_BLOCKETTE.offset());
        short lastBlocketteStart = 0;
        int lastBlocketteId = 0;
        while (nextBlocketteStart != 0) {
            lastBlocketteStart = nextBlocketteStart;
            lastBlocketteId = this.buffer.getShort(lastBlocketteStart + DataBlockette.BLOCKETTE_ID.offset());
            if (lastBlocketteId == 1001) {
                return false;
            }
            nextBlocketteStart = this.buffer.getShort(nextBlocketteStart + DataBlockette.NEXT_BLOCKETTE.offset());
        }
        switch (lastBlocketteId) {
            case 100: {
                lastBlocketteSize = (short)Blockette100.totalSize();
                break;
            }
            case 1000: {
                lastBlocketteSize = (short)Blockette1000.totalSize();
                break;
            }
            default: {
                return false;
            }
        }
        short beginData = this.buffer.getShort(FixedHeader.BEGIN_DATA.offset());
        int available = beginData - (lastBlocketteStart + lastBlocketteSize);
        if (available >= Blockette1001.totalSize()) {
            int start = lastBlocketteStart + lastBlocketteSize;
            this.buffer.putShort(lastBlocketteStart + DataBlockette.NEXT_BLOCKETTE.offset(), (short)start);
            this.buffer.putShort(start + Blockette1001.BLOCKETTE_ID.offset(), (short)1001);
            this.buffer.putShort(start + Blockette1001.NEXT_BLOCKETTE.offset(), (short)0);
            this.buffer.put(start + Blockette1001.TIMING_QUALITY.offset(), (byte)0);
            this.buffer.put(start + Blockette1001.TIME_MICROSECOND.offset(), (byte)0);
            this.buffer.put(start + Blockette1001.DATA_FRAMES.offset(), (byte)0);
            this.beginBlockette1001 = start;
            return true;
        }
        return false;
    }

    public int hashCode() {
        int result = 1;
        for (byte element : this.buffer.array()) {
            result = 31 * result + element;
        }
        return result;
    }

    public boolean equals(Object other) {
        if (this == other) {
            return true;
        }
        if (other == null) {
            return false;
        }
        if (this.getClass() != other.getClass()) {
            return false;
        }
        return Arrays.equals(this.buffer.array(), ((Record)other).buffer.array());
    }

    public String toString() {
        if (this.getCodec() == EncodingFormat.ASCII) {
            switch (this.getSampleCount()) {
                case 0: {
                    return "#" + this.getSequenceString() + ": " + this.getStationId() + ", " + this.getChannelId() + ", " + this.getStartTime().toLogString();
                }
                case 1: {
                    return "#" + this.getSequenceString() + ": " + this.getStationId() + ", " + this.getChannelId() + ", " + this.getStartTime().toLogString() + ", " + this.getSampleCount() + " character";
                }
            }
            return "#" + this.getSequenceString() + ": " + this.getStationId() + ", " + this.getChannelId() + ", " + this.getStartTime().toLogString() + ", " + this.getSampleCount() + " characters";
        }
        return "#" + this.getSequenceString() + ": " + this.getStationId() + ", " + this.getChannelId() + ", " + this.getStartTime() + ", " + this.getSampleCount() + " samples";
    }

    @Override
    public int compareTo(Record other) {
        assert (other != null);
        if (!this.getNetworkId().equals(other.getNetworkId())) {
            return this.getNetworkId().compareTo(other.getNetworkId());
        }
        if (!this.getLocationId().equals(other.getLocationId())) {
            return this.getLocationId().compareTo(other.getLocationId());
        }
        if (!this.getStationId().equals(other.getStationId())) {
            return this.getStationId().compareTo(other.getStationId());
        }
        if (!this.getChannelId().equals(other.getChannelId())) {
            return this.getChannelId().compareTo(other.getChannelId());
        }
        return this.getStartTime().compareTo(other.getStartTime());
    }

    public byte[] digest(MessageDigest digestAlgorithm) {
        if (digestAlgorithm == null) {
            return null;
        }
        digestAlgorithm.update(this.buffer.array(), 0, this.size());
        return digestAlgorithm.digest();
    }

    public boolean beginsWithMagicNumber() {
        block3: for (int i = 0; i < FixedHeader.SEQUENCE_NO.length(); ++i) {
            byte number = this.buffer.get(FixedHeader.SEQUENCE_NO.offset() + i);
            switch (number) {
                case 0: 
                case 32: 
                case 48: 
                case 49: 
                case 50: 
                case 51: 
                case 52: 
                case 53: 
                case 54: 
                case 55: 
                case 56: 
                case 57: {
                    continue block3;
                }
                default: {
                    return false;
                }
            }
        }
        byte indicator = this.buffer.get(FixedHeader.DATA_INDICATOR.offset());
        if (indicator != 68 && indicator != 82 && indicator != 81 && indicator != 77) {
            return false;
        }
        return this.buffer.get(FixedHeader.SPACE.offset()) == 32;
    }

    public int read(InputStream inStream) throws IntegrityException, EOFException, IllegalStateException, IOException {
        if (!this.buffer.hasArray()) {
            throw new IllegalStateException("The internal ByteBuffer of the 'Record' is not backed by an accessible (i.e. writable) byte array.");
        }
        if (this.buffer.capacity() < 64) {
            throw new IllegalStateException("The capacity of the internal ByteBuffer is not high enough to read even a minimal miniSEED header (64 bytes).");
        }
        this.buffer.clear();
        this.beginBlockette100 = 0;
        this.beginBlockette1000 = 0;
        this.beginBlockette1001 = 0;
        int count = inStream.read(this.buffer.array(), 0, 64);
        if (count == -1) {
            return -1;
        }
        this.buffer.position(count);
        if (count != 64) {
            throw new EOFException("Could not completely read the first part of the miniSEED header. (Only got " + count + " of expected " + 64 + " bytes .)");
        }
        if (!this.beginsWithMagicNumber()) {
            throw new IntegrityException("Invalid file format! MiniSEED record does not begin with a six digit number followed by the character 'D', 'R', 'Q' or 'M' and a space character (e.g. '123456D').");
        }
        while (this.beginBlockette1000 == 0) {
            try {
                if (this.findBlockette(1000, count) != -1) continue;
                throw new IntegrityException("MiniSEED record does not contain the mandatory blockette #1000.");
            }
            catch (IndexOutOfBoundsException e) {
                int doubledSize = count * 2;
                if (this.buffer.capacity() < doubledSize) {
                    ByteBuffer largerBuffer = ByteBuffer.allocate(doubledSize);
                    largerBuffer.order(this.buffer.order());
                    largerBuffer.put(this.buffer.array(), 0, count);
                    this.buffer = largerBuffer;
                }
                if ((count += inStream.read(this.buffer.array(), count, doubledSize - count)) != doubledSize) {
                    throw new EOFException("Could not completely read the miniSEED record. (Only got the first " + count + " of expected " + doubledSize + " bytes.)");
                }
                this.buffer.position(count);
            }
        }
        this.buffer.order(this.getByteOrder());
        this.fixByteOrder();
        if (this.size() < 64) {
            if (this.size() == 0) {
                throw new IntegrityException("Invalid miniSEED! There is no \"record size\" information given.");
            }
            throw new IntegrityException("Invalid miniSEED! The smallest possible miniSEED record size is 64 bytes but a size of " + this.size() + " bytes was given.");
        }
        if (this.buffer.capacity() < this.size()) {
            ByteBuffer newBuffer = ByteBuffer.allocate(this.size());
            newBuffer.order(this.buffer.order());
            newBuffer.put(this.buffer.array(), 0, count);
            this.buffer = newBuffer;
        }
        if ((count += inStream.read(this.buffer.array(), count, this.size() - count)) != this.size()) {
            throw new EOFException("Could not completely read the miniSEED record. (Only got the first " + count + " of expected " + this.size() + " bytes.)");
        }
        this.buffer.rewind();
        return count;
    }

    public void write(OutputStream outStream) throws IOException, IntegrityException {
        if (!this.beginsWithMagicNumber()) {
            throw new IntegrityException("MiniSEED record does not begin with a six digit number followed by the character 'D', 'R', 'Q' or 'M' and a space character (e.g. '123456D').");
        }
        outStream.write(this.buffer.array(), 0, this.size());
    }

    public int size() {
        int exponent = 0xFF & this.buffer.get(this.findBlockette(1000, this.buffer.capacity()) + Blockette1000.RECORD_LENGTH.offset());
        if (exponent == 0) {
            return 0;
        }
        return 1 << exponent;
    }

    public boolean hasBlockette(int id) {
        return this.findBlockette(id, this.buffer.capacity()) > 0;
    }

    public String getSequenceString() {
        String PADDING = "      ";
        try {
            String buf = Record.sanitizeText(new String(this.buffer.array(), FixedHeader.SEQUENCE_NO.offset(), FixedHeader.SEQUENCE_NO.length(), StandardCharsets.ISO_8859_1));
            return (buf + "      ").substring(0, 6);
        }
        catch (Exception e) {
            return "      ";
        }
    }

    public int getSequenceNumber() {
        try {
            return Integer.parseInt(new String(this.buffer.array(), FixedHeader.SEQUENCE_NO.offset(), FixedHeader.SEQUENCE_NO.length(), StandardCharsets.ISO_8859_1));
        }
        catch (Exception e) {
            return -1;
        }
    }

    public Record setSequenceNumber(int number) throws IllegalArgumentException {
        if (number < 1 || number > 999999) {
            throw new IllegalArgumentException("The \"sequence number\" must be a positive number in the range from 1 to 999999 (but was " + number + ").");
        }
        String text = StringUtils.SIX_DIGITS.format(number);
        this.pokeText(FixedHeader.SEQUENCE_NO.offset(), FixedHeader.SEQUENCE_NO.length(), text);
        return this;
    }

    public String getStationId() {
        String station = new String(this.buffer.array(), FixedHeader.STATION_ID.offset(), FixedHeader.STATION_ID.length(), StandardCharsets.ISO_8859_1);
        return Record.sanitizeText(station);
    }

    public Record setStationId(String station) throws IllegalArgumentException {
        if (station.length() > FixedHeader.STATION_ID.length()) {
            throw new IllegalArgumentException("The \"station id\" is longer than the allowed " + FixedHeader.STATION_ID.length() + " characters.");
        }
        this.pokeText(FixedHeader.STATION_ID.offset(), FixedHeader.STATION_ID.length(), station);
        return this;
    }

    public String getLocationId() {
        String location = new String(this.buffer.array(), FixedHeader.LOCATION_ID.offset(), FixedHeader.LOCATION_ID.length(), StandardCharsets.ISO_8859_1);
        return Record.sanitizeText(location);
    }

    public Record setLocationId(String location) throws IllegalArgumentException {
        if (location.length() > FixedHeader.LOCATION_ID.length()) {
            throw new IllegalArgumentException("The \"location id\" is longer than the allowed " + FixedHeader.LOCATION_ID.length() + " characters.");
        }
        this.pokeText(FixedHeader.LOCATION_ID.offset(), FixedHeader.LOCATION_ID.length(), location);
        return this;
    }

    public String getChannelId() {
        String channel = new String(this.buffer.array(), FixedHeader.CHANNEL_ID.offset(), FixedHeader.CHANNEL_ID.length(), StandardCharsets.ISO_8859_1);
        return Record.sanitizeText(channel);
    }

    public Record setChannelId(String channel) throws IllegalArgumentException {
        if (channel.length() > FixedHeader.CHANNEL_ID.length()) {
            throw new IllegalArgumentException("The \"channel id\" is longer than the allowed " + FixedHeader.LOCATION_ID.length() + " characters.");
        }
        this.pokeText(FixedHeader.CHANNEL_ID.offset(), FixedHeader.CHANNEL_ID.length(), channel);
        return this;
    }

    public String getNetworkId() {
        String network = new String(this.buffer.array(), FixedHeader.NETWORK_ID.offset(), FixedHeader.NETWORK_ID.length(), StandardCharsets.ISO_8859_1);
        return Record.sanitizeText(network);
    }

    public Record setNetworkId(String network) throws IllegalArgumentException {
        if (network.length() > FixedHeader.NETWORK_ID.length()) {
            throw new IllegalArgumentException("The \"network id\" is longer than the allowed " + FixedHeader.LOCATION_ID.length() + " characters.");
        }
        this.pokeText(FixedHeader.NETWORK_ID.offset(), FixedHeader.NETWORK_ID.length(), network);
        return this;
    }

    public TimeMoment getStartTime() {
        int usecCorrection = 0;
        short year = this.buffer.getShort(FixedHeader.TIME_YEAR.offset());
        short doy = this.buffer.getShort(FixedHeader.TIME_DAYOFYEAR.offset());
        byte hour = this.buffer.get(FixedHeader.TIME_HOUR.offset());
        byte min = this.buffer.get(FixedHeader.TIME_MINUTE.offset());
        byte sec = this.buffer.get(FixedHeader.TIME_SECOND.offset());
        short tick = this.buffer.getShort(FixedHeader.TIME_TICKS.offset());
        if (year == 0 && doy == 0 && hour == 0 && min == 0 && sec == 0 && tick == 0) {
            return null;
        }
        assert (year >= 0) : "Bad year (" + year + ") in fixed header";
        assert (doy >= 1 && doy <= 366) : "Bad day-of-year (" + doy + ") in fixed header!";
        assert (hour >= 0 && hour <= 23) : "Bad hour (" + hour + ") in fixed header!";
        assert (min >= 0 && min <= 59) : "Bad minutes (" + min + ") in fixed header!";
        assert (sec >= 0 && sec <= 60) : "Bad seconds (" + sec + ") in fixed header!";
        assert (tick >= 0 && tick <= 9999) : "Bad ticks (" + tick + ") in fixed header!";
        int activity = 0xFF & this.buffer.get(FixedHeader.ACTIVITY_FLAG.offset());
        int tickCorrection = (activity & 2) == 0 ? this.buffer.getInt(FixedHeader.TICK_CORRECTION.offset()) : 0;
        if (this.hasBlockette(1001)) {
            usecCorrection = this.buffer.get(this.findBlockette(1001, this.buffer.capacity()) + Blockette1001.TIME_MICROSECOND.offset());
        }
        int usec = (tick + tickCorrection) * 100 + usecCorrection;
        return new TimeMoment.Builder().dayOfYear(year, doy).time(hour, min, sec).microsec(usec).build();
    }

    public Record setStartTime(TimeMoment time) {
        return this.setStartTime(time, 0, true);
    }

    public Record setStartTime(TimeMoment time, int correction, boolean doApply) throws UnsupportedException {
        int year = time.getYear();
        int doy = time.getDayOfYear();
        int hour = time.getHour();
        int min = time.getMinute();
        int sec = time.getSecond();
        long frac = time.getNanoSecond();
        assert (year >= 0 && year <= 65535) : "Bad year value (" + year + ") for fixed header!";
        assert (doy >= 1 && doy <= 366) : "Bad doy value (" + doy + ") for fixed header!";
        assert (hour >= 0 && hour <= 23) : "Bad hour value (" + hour + ") for fixed header!";
        assert (min >= 0 && min <= 59) : "Bad min value (" + min + ") for fixed header!";
        assert (sec >= 0 && sec <= 60) : "Bad sec value (" + sec + ") for fixed header!";
        int tick = (int)(frac / 100000L);
        int usec = (int)Math.round((double)(frac % 100000L) / 1000.0);
        assert (tick >= 0 && tick <= 9999) : "Bad tick value (" + tick + ") for fixed header!";
        assert (usec >= -50 && usec <= 99) : "Bad microsecond count (" + usec + ") for blockette #1001!";
        this.buffer.putShort(FixedHeader.TIME_YEAR.offset(), (short)year);
        this.buffer.putShort(FixedHeader.TIME_DAYOFYEAR.offset(), (short)doy);
        this.buffer.put(FixedHeader.TIME_HOUR.offset(), (byte)hour);
        this.buffer.put(FixedHeader.TIME_MINUTE.offset(), (byte)min);
        this.buffer.put(FixedHeader.TIME_SECOND.offset(), (byte)sec);
        this.buffer.putShort(FixedHeader.TIME_TICKS.offset(), (short)tick);
        byte oldFlag = this.buffer.get(FixedHeader.ACTIVITY_FLAG.offset());
        byte newFlag = (byte)(doApply ? oldFlag & 0xFD : oldFlag | 2);
        this.buffer.put(FixedHeader.ACTIVITY_FLAG.offset(), newFlag);
        this.buffer.putInt(FixedHeader.TICK_CORRECTION.offset(), correction);
        if (this.hasBlockette(1001) || usec != 0 && this.injectBlockette1001()) {
            this.buffer.put(this.findBlockette(1001, this.buffer.capacity()) + Blockette1001.TIME_MICROSECOND.offset(), (byte)usec);
        } else if (usec != 0) {
            throw new UnsupportedException("Failed to add a new blockette #1001 to the miniSEED record and hence cannot store start time with microsecond precision ('" + time + "').");
        }
        return this;
    }

    public TimeMoment getStopTime() {
        if (this.getStartTime() == null) {
            return null;
        }
        if (this.getSampleCount() == 0) {
            return this.getStartTime();
        }
        return this.getTimeAt(this.getSampleCount() - 1);
    }

    public EncodingFormat getCodec() {
        byte code = this.buffer.get(this.findBlockette(1000, this.buffer.capacity()) + Blockette1000.DATA_FORMAT.offset());
        for (EncodingFormat codec : EncodingFormat.values()) {
            if (codec.getCode() != code) continue;
            return codec;
        }
        return null;
    }

    public Record setCodec(EncodingFormat encoding) {
        if (encoding == null) {
            throw new NullPointerException("The given miniSEED encoding is 'null'!");
        }
        this.buffer.put(this.findBlockette(1000, this.buffer.capacity()) + Blockette1000.DATA_FORMAT.offset(), encoding.getCode());
        return this;
    }

    public int getSampleCount() {
        return this.buffer.getShort(FixedHeader.SAMPLE_COUNT.offset());
    }

    Record setSampleCount(int samples) throws IllegalArgumentException {
        if (samples < 0) {
            throw new IllegalArgumentException("Negative number of samples.");
        }
        this.buffer.putShort(FixedHeader.SAMPLE_COUNT.offset(), (short)samples);
        return this;
    }

    public double getSamplePeriod() throws IntegrityException {
        double samplePeriod = -1.0;
        if (this.hasBlockette(100)) {
            samplePeriod = 1.0 / (double)this.buffer.getFloat(this.findBlockette(100, this.buffer.capacity()) + Blockette100.SAMPLE_RATE.offset());
            assert (samplePeriod >= 0.0) : "Sample period in blockette #100 is negative!";
        }
        if (EncodingFormat.ASCII == this.getCodec()) {
            return 0.0;
        }
        if (samplePeriod <= 0.0) {
            short factor = this.buffer.getShort(FixedHeader.SAMPLE_FACTOR.offset());
            short multiplier = this.buffer.getShort(FixedHeader.SAMPLE_MULTIPLIER.offset());
            assert (factor != 0) : "Sample rate factor in fixed header is 0!";
            assert (multiplier != 0) : "Sample rate multiplier in fixed header is 0!";
            if (factor > 0 && multiplier == 1) {
                samplePeriod = 1.0 / (double)factor;
            } else if (factor > 0 && multiplier > 0) {
                samplePeriod = 1.0 / ((double)factor * (double)multiplier);
            } else if (factor > 0 && multiplier < 0) {
                samplePeriod = -1.0 * (double)multiplier / (double)factor;
            } else if (factor < 0 && multiplier > 0) {
                samplePeriod = -1.0 * (double)factor / (double)multiplier;
            } else if (factor < 0 && multiplier < 0) {
                samplePeriod = (double)factor * (double)multiplier;
            } else {
                throw new IntegrityException("Invalid sampling rate definition. (factor=" + factor + ", multiplier=" + multiplier + ")");
            }
        }
        return samplePeriod;
    }

    public Record setSamplePeriod(double samplePeriod) {
        assert (samplePeriod > 0.0) : "Sample period must be positive!";
        short sampleRate = (short)(1.0 / samplePeriod);
        this.buffer.putShort(FixedHeader.SAMPLE_FACTOR.offset(), sampleRate);
        this.buffer.putShort(FixedHeader.SAMPLE_MULTIPLIER.offset(), (short)1);
        if (this.hasBlockette(100)) {
            this.buffer.putFloat(this.findBlockette(100, this.buffer.capacity()) + Blockette100.SAMPLE_RATE.offset(), sampleRate);
        }
        return this;
    }

    public double getSampleRate() throws IntegrityException {
        double sampleRate = -1.0;
        if (this.hasBlockette(100)) {
            sampleRate = this.buffer.getFloat(this.findBlockette(100, this.buffer.capacity()) + Blockette100.SAMPLE_RATE.offset());
            assert (sampleRate >= 0.0) : "Sample rate in blockette #100 is negative!";
        }
        if (EncodingFormat.ASCII == this.getCodec()) {
            return 0.0;
        }
        if (sampleRate <= 0.0) {
            short factor = this.buffer.getShort(FixedHeader.SAMPLE_FACTOR.offset());
            short multiplier = this.buffer.getShort(FixedHeader.SAMPLE_MULTIPLIER.offset());
            assert (factor != 0) : "Sample rate factor in fixed header is 0!";
            assert (multiplier != 0) : "Sample rate multiplier in fixed header is 0!";
            if (factor > 0 && multiplier == 1) {
                sampleRate = factor;
            } else if (factor > 0 && multiplier > 0) {
                sampleRate = (double)factor * (double)multiplier;
            } else if (factor > 0 && multiplier < 0) {
                sampleRate = -1.0 * (double)factor / (double)multiplier;
            } else if (factor < 0 && multiplier > 0) {
                sampleRate = -1.0 * (double)multiplier / (double)factor;
            } else if (factor < 0 && multiplier < 0) {
                sampleRate = 1.0 / ((double)multiplier * (double)factor);
            } else {
                throw new IntegrityException("Invalid sampling rate definition. (factor=" + factor + ", multiplier=" + multiplier + ")");
            }
        }
        return sampleRate;
    }

    public TimeMoment getTimeAt(long index) throws IntegrityException {
        TimeSpan offset;
        short factor = this.buffer.getShort(FixedHeader.SAMPLE_FACTOR.offset());
        short multiplier = this.buffer.getShort(FixedHeader.SAMPLE_MULTIPLIER.offset());
        if (factor > 0 && multiplier == 1) {
            offset = new TimeSpan.Builder().addSeconds((double)index / (double)factor).build();
        } else if (factor > 0 && multiplier > 0) {
            offset = new TimeSpan.Builder().addSeconds((double)index / (double)(factor * multiplier)).build();
        } else if (factor > 0 && multiplier < 0) {
            offset = new TimeSpan.Builder().addSeconds((double)index * (double)(-multiplier) / (double)factor).build();
        } else if (factor < 0 && multiplier > 0) {
            offset = new TimeSpan.Builder().addSeconds((double)index * (double)(-factor * multiplier)).build();
        } else if (factor < 0 && multiplier < 0) {
            offset = new TimeSpan.Builder().addSeconds((double)index * (double)factor / (double)multiplier).build();
        } else if (EncodingFormat.ASCII == this.getCodec()) {
            offset = TimeSpan.ZERO;
        } else {
            throw new IntegrityException("The 'sample factor' and / or 'sample multiplier' are 0, which is undefined. Cannot determine the sample rate of this miniSEED record.");
        }
        if (index > 0L) {
            return TimeSpan.add(this.getStartTime(), offset);
        }
        return TimeSpan.subtract(this.getStartTime(), offset);
    }

    public long getLastIndexBefore(TimeMoment time, boolean including) {
        assert (time != null) : "Argument 'time' must not be null.";
        double index = TimeSpan.diff(time, this.getStartTime()).divide(new TimeSpan.Builder().addSeconds(this.getSamplePeriod()).build());
        if (index > 9.223372036854776E18 || index < -9.223372036854776E18) {
            throw new IndexOutOfBoundsException("Index of the next sample before time " + time + " is #" + index + ", which cannot be represented by a 'long' variable!");
        }
        if (this.getStartTime().before(time)) {
            if (!including && index == Math.floor(index)) {
                return Math.round(Math.floor(index)) - 1L;
            }
            return Math.round(Math.floor(index));
        }
        if (!including && index == Math.floor(index)) {
            return Math.round(Math.floor(-index)) - 1L;
        }
        return Math.round(Math.floor(-index));
    }

    public long getFirstIndexAfter(TimeMoment time, boolean including) {
        assert (time != null) : "Argument 'time' must not be null.";
        double index = TimeSpan.diff(time, this.getStartTime()).divide(new TimeSpan.Builder().addSeconds(this.getSamplePeriod()).build());
        if (index > 9.223372036854776E18 || index < -9.223372036854776E18) {
            throw new IndexOutOfBoundsException("Index of the next sample after time " + time + " is #" + index + ", which cannot be represented by a 'long' variable!");
        }
        if (this.getStartTime().before(time)) {
            if (!including && index == Math.ceil(index)) {
                return Math.round(Math.ceil(index)) + 1L;
            }
            return Math.round(Math.ceil(index));
        }
        if (!including && index == Math.ceil(index)) {
            return Math.round(Math.ceil(-index)) + 1L;
        }
        return Math.round(Math.ceil(-index));
    }

    public long getClosestIndex(TimeMoment time) {
        assert (time != null) : "Argument 'time' must not be null.";
        double index = TimeSpan.diff(time, this.getStartTime()).divide(new TimeSpan.Builder().addSeconds(this.getSamplePeriod()).build());
        if (index > 9.223372036854776E18 || index < -9.223372036854776E18) {
            throw new IndexOutOfBoundsException("Index of the sample closest to time " + time + " is #" + index + ", which cannot be represented by a 'long' variable!");
        }
        if (this.getStartTime().before(time)) {
            return Math.round(index);
        }
        return Math.round(-index);
    }

    public ByteOrder getByteOrder() throws IntegrityException {
        if (this.hasBlockette(1000)) {
            byte flag = this.buffer.get(this.findBlockette(1000, this.buffer.capacity()) + Blockette1000.BYTE_ORDER.offset());
            if (flag == 1) {
                return ByteOrder.BIG_ENDIAN;
            }
            if (flag == 0) {
                return ByteOrder.LITTLE_ENDIAN;
            }
            throw new IntegrityException("Illegal \"byte order\" flag (" + flag + "). Only 1 (big endian) and 0 (little endian) are allowed.");
        }
        return this.buffer.order();
    }

    private void setByteOrder(ByteOrder byteOrder) throws IllegalArgumentException {
        if (byteOrder == ByteOrder.BIG_ENDIAN) {
            this.buffer.put(this.findBlockette(1000, this.buffer.capacity()) + Blockette1000.BYTE_ORDER.offset(), (byte)1);
        } else if (byteOrder == ByteOrder.LITTLE_ENDIAN) {
            this.buffer.put(this.findBlockette(1000, this.buffer.capacity()) + Blockette1000.BYTE_ORDER.offset(), (byte)0);
        } else {
            throw new IllegalArgumentException("Unknown byte order (" + byteOrder + "). Only " + ByteOrder.BIG_ENDIAN + " and " + ByteOrder.LITTLE_ENDIAN + " are allowed.");
        }
        this.buffer.order(byteOrder);
    }

    public String getByteOrderString() {
        try {
            ByteOrder order = this.getByteOrder();
            if (order.equals(ByteOrder.LITTLE_ENDIAN)) {
                return "little endian";
            }
            if (order.equals(ByteOrder.BIG_ENDIAN)) {
                return "big endian";
            }
            return "UNKNOWN BYTE ORDER";
        }
        catch (IntegrityException e) {
            return "ILLEGAL BYTE ORDER";
        }
    }

    private void fixByteOrder() {
        short originalYear = this.buffer.getShort(FixedHeader.TIME_YEAR.offset());
        if (originalYear >= 1900 && originalYear <= 2055 || originalYear == 0) {
            return;
        }
        short swappedYear = Short.reverseBytes(originalYear);
        if (swappedYear >= 1900 && swappedYear <= 2055 || swappedYear == 0) {
            IOUtils.swapShort(this.buffer.array(), FixedHeader.TIME_YEAR.offset());
            IOUtils.swapShort(this.buffer.array(), FixedHeader.TIME_DAYOFYEAR.offset());
            IOUtils.swapShort(this.buffer.array(), FixedHeader.TIME_TICKS.offset());
            IOUtils.swapShort(this.buffer.array(), FixedHeader.SAMPLE_COUNT.offset());
            IOUtils.swapShort(this.buffer.array(), FixedHeader.SAMPLE_FACTOR.offset());
            IOUtils.swapShort(this.buffer.array(), FixedHeader.SAMPLE_MULTIPLIER.offset());
            IOUtils.swapInt(this.buffer.array(), FixedHeader.TICK_CORRECTION.offset());
            IOUtils.swapShort(this.buffer.array(), FixedHeader.BEGIN_DATA.offset());
            IOUtils.swapShort(this.buffer.array(), FixedHeader.BEGIN_BLOCKETTE.offset());
            short beginBlockette = this.buffer.getShort(FixedHeader.BEGIN_BLOCKETTE.offset());
            while (beginBlockette > 0) {
                IOUtils.swapShort(this.buffer.array(), beginBlockette + DataBlockette.BLOCKETTE_ID.offset());
                IOUtils.swapShort(this.buffer.array(), beginBlockette + DataBlockette.NEXT_BLOCKETTE.offset());
                switch (this.buffer.getShort(beginBlockette + DataBlockette.BLOCKETTE_ID.offset())) {
                    case 100: {
                        IOUtils.swapInt(this.buffer.array(), beginBlockette + Blockette100.SAMPLE_RATE.offset());
                        break;
                    }
                }
                beginBlockette = this.buffer.getShort(beginBlockette + DataBlockette.NEXT_BLOCKETTE.offset());
            }
        } else {
            throw new IntegrityException("Strange recording year in miniSEED record (" + originalYear + "). Expected something between 1900 and 2055.");
        }
    }

    private int getBeginOfData() {
        short begin = this.buffer.getShort(FixedHeader.BEGIN_DATA.offset());
        if (begin <= 0) {
            return 64;
        }
        return begin;
    }

    public double[] getSampleData() {
        return this.getSampleData(null, 0, 0, this.getSampleCount());
    }

    public double[] getSampleData(double[] dest, int destPos, int srcPos, int length) throws IllegalArgumentException, UnsupportedException, IntegrityException {
        MiniseedCodec codec;
        assert (length >= 0) : "Cannot obtain a negative number of values.";
        assert (srcPos >= 0) : "Index of first sample to decode must be positive.";
        assert (srcPos + length <= this.getSampleCount()) : "There are not enough samples in the record.";
        double[] allSamples = new double[this.getSampleCount()];
        EncodingFormat format = this.getCodec();
        if (format == null) {
            throw new UnsupportedException("Data encoding format #" + this.buffer.get(this.findBlockette(1000, this.buffer.capacity()) + Blockette1000.DATA_FORMAT.offset()) + " is unknown! Currently, only 'Steim-1', 'Steim-2', '32-bit plain integer', 'IEEE float' and 'IEEE double floating point' are supported.");
        }
        switch (format) {
            case INT_32: {
                codec = new IntegerCodec();
                break;
            }
            case IEEE_FLOAT: {
                codec = new IeeeFloatCodec();
                break;
            }
            case IEEE_DOUBLE: {
                codec = new IeeeDoubleCodec();
                break;
            }
            case STEIM_1: {
                codec = new SteimOneCodec();
                break;
            }
            case STEIM_2: {
                codec = new SteimTwoCodec();
                break;
            }
            default: {
                throw new UnsupportedException("Unsupported data encoding ('" + (Object)((Object)format) + "'). Currently, only 'Steim-1', 'Steim-2', '32 bit plain integer', 'IEEE float' and 'IEEE double floating point' are supported.");
            }
        }
        int count = codec.decodeSamples(this.buffer, this.getBeginOfData(), allSamples, 0, allSamples.length);
        if (count != allSamples.length) {
            throw new IntegrityException("Decoded " + count + " values but expected " + length + " samples!");
        }
        if (dest == null) {
            if (destPos == 0 && length == allSamples.length) {
                return allSamples;
            }
            double[] result = new double[destPos + length];
            System.arraycopy(allSamples, srcPos, result, destPos, length);
            return result;
        }
        System.arraycopy(allSamples, srcPos, dest, destPos, length);
        return dest;
    }

    public byte[] getRawData() {
        return this.getRawData(null, 0, 0, 0);
    }

    public byte[] getRawData(byte[] dest, int destPos, int srcPos, int length) {
        int numOfBytes;
        assert (srcPos >= 0) : "Source index must be positive.";
        assert (destPos >= 0) : "Destination index must be positive.";
        this.buffer.position(this.getBeginOfData() + srcPos);
        if (length > 0) {
            assert (length <= this.buffer.remaining()) : "Requested more data than available in record.";
            numOfBytes = length;
        } else {
            numOfBytes = this.buffer.remaining();
        }
        if (dest != null) {
            this.buffer.get(dest, destPos, numOfBytes);
            return dest;
        }
        byte[] result = new byte[numOfBytes];
        this.buffer.get(result, 0, numOfBytes);
        return result;
    }

    public int putSampleData(double[] src, int srcPos, double bias) throws UnsupportedException {
        return this.putSampleData(src, srcPos, src.length - srcPos, bias);
    }

    public int putSampleData(double[] src, int srcPos, int maxLength, double bias) throws UnsupportedException {
        MiniseedCodec codec;
        EncodingFormat format = this.getCodec();
        if (src.length < srcPos + maxLength) {
            throw new IllegalArgumentException("There are not enough samples in the array!");
        }
        if (format == null) {
            throw new UnsupportedException("Data encoding format #" + this.buffer.get(this.findBlockette(1000, this.buffer.capacity()) + Blockette1000.DATA_FORMAT.offset()) + " is unknown! Currently, only 'Steim-1', 'Steim-2', '32-bit plain integer', 'IEEE float' and 'IEEE double floating point' are supported.");
        }
        switch (this.getCodec()) {
            case INT_32: {
                codec = new IntegerCodec();
                break;
            }
            case IEEE_FLOAT: {
                codec = new IeeeFloatCodec();
                break;
            }
            case IEEE_DOUBLE: {
                codec = new IeeeDoubleCodec();
                break;
            }
            case STEIM_1: {
                codec = new SteimOneCodec();
                break;
            }
            case STEIM_2: {
                codec = new SteimTwoCodec();
                break;
            }
            default: {
                throw new UnsupportedException("Unsupported data encoding (format #" + (Object)((Object)format) + "). Currently, only 'Steim-1', 'Steim-2', '32-bit plain integer', 'IEEE float' and 'IEEE double floating point' are supported.");
            }
        }
        int count = codec.encodeSamples(src, srcPos, this.buffer, this.getBeginOfData(), maxLength, bias);
        this.setSampleCount(count);
        return count;
    }

    protected void clearData() {
        for (int i = this.getBeginOfData(); i < this.size(); ++i) {
            this.buffer.put(i, (byte)0);
        }
        this.setSampleCount(0);
    }

    public double firstSampleValue() {
        MiniseedCodec codec;
        EncodingFormat format = this.getCodec();
        if (format == null) {
            throw new UnsupportedException("Data encoding format #" + this.buffer.get(this.findBlockette(1000, this.buffer.capacity()) + Blockette1000.DATA_FORMAT.offset()) + " is unknown! Currently, only 'Steim-1', 'Steim-2', '32-bit plain integer', 'IEEE float' and 'IEEE double floating point' are supported.");
        }
        switch (this.getCodec()) {
            case INT_32: {
                codec = new IntegerCodec();
                break;
            }
            case IEEE_FLOAT: {
                codec = new IeeeFloatCodec();
                break;
            }
            case IEEE_DOUBLE: {
                codec = new IeeeDoubleCodec();
                break;
            }
            case STEIM_1: {
                codec = new SteimOneCodec();
                break;
            }
            case STEIM_2: {
                codec = new SteimTwoCodec();
                break;
            }
            default: {
                throw new UnsupportedException("Unsupported data encoding (format #" + (Object)((Object)format) + "). Currently, only 'Steim-1', 'Steim-2', '32-bit plain integer', 'IEEE float' and 'IEEE double floating point' are supported.");
            }
        }
        return codec.getFirstSample(this.buffer, this.getBeginOfData());
    }

    public double lastSampleValue() {
        MiniseedCodec codec;
        EncodingFormat format = this.getCodec();
        if (format == null) {
            throw new UnsupportedException("Data encoding format #" + this.buffer.get(this.findBlockette(1000, this.buffer.capacity()) + Blockette1000.DATA_FORMAT.offset()) + " is unknown! Currently, only 'Steim-1', 'Steim-2', '32-bit plain integer', 'IEEE float' and 'IEEE double floating point' are supported.");
        }
        switch (this.getCodec()) {
            case INT_32: {
                codec = new IntegerCodec();
                break;
            }
            case IEEE_FLOAT: {
                codec = new IeeeFloatCodec();
                break;
            }
            case IEEE_DOUBLE: {
                codec = new IeeeDoubleCodec();
                break;
            }
            case STEIM_1: {
                codec = new SteimOneCodec();
                break;
            }
            case STEIM_2: {
                codec = new SteimTwoCodec();
                break;
            }
            default: {
                throw new UnsupportedException("Unsupported data encoding (format #" + (Object)((Object)format) + "). Currently, only 'Steim-1', 'Steim-2', '32-bit plain integer', 'IEEE float' and 'IEEE double floating point' are supported.");
            }
        }
        return codec.getLastSample(this.buffer, this.getBeginOfData(), this.getSampleCount());
    }

    public int getActivityFlag() {
        return 0xFF & this.buffer.get(FixedHeader.ACTIVITY_FLAG.offset());
    }

    public String getActivityInfo() {
        int flag = this.getActivityFlag();
        int tickCorrection = this.buffer.getInt(FixedHeader.TICK_CORRECTION.offset());
        if (flag == 0 && tickCorrection == 0) {
            return "0x" + StringUtils.toHexString(flag, 2);
        }
        StringBuilder info = new StringBuilder();
        this.appendBitMaskInfo(info, flag, 1, "calibration signals present");
        if (tickCorrection != 0) {
            if (info.length() > 0) {
                info.append(", ");
            }
            if ((flag & 2) == 2) {
                info.append("contains a time correction of ").append((double)tickCorrection / 10000.0).append(" seconds");
            } else {
                info.append("time correction of ").append((double)tickCorrection / 10000.0).append(" seconds will be applied");
            }
        } else {
            this.appendBitMaskInfo(info, flag, 2, "time correction was applied");
        }
        this.appendBitMaskInfo(info, flag, 4, "beginning of an event (station trigger)");
        this.appendBitMaskInfo(info, flag, 8, "end of the event (station de-triggers)");
        this.appendBitMaskInfo(info, flag, 16, "positive leap second happened during this record");
        this.appendBitMaskInfo(info, flag, 32, "negative leap second happened during this record");
        this.appendBitMaskInfo(info, flag, 64, "event in progress");
        return "0x" + StringUtils.toHexString(flag, 2) + " (" + info + ")";
    }

    public int getIoAndClockFlag() {
        return 0xFF & this.buffer.get(FixedHeader.IO_AND_CLOCK_FLAG.offset());
    }

    public String getIoAndClockInfo() {
        int flag = this.getIoAndClockFlag();
        if (flag == 0) {
            return "0x" + StringUtils.toHexString(flag, 2);
        }
        StringBuilder info = new StringBuilder();
        this.appendBitMaskInfo(info, flag, 1, "station volume parity error");
        this.appendBitMaskInfo(info, flag, 2, "long record read");
        this.appendBitMaskInfo(info, flag, 4, "short record read");
        this.appendBitMaskInfo(info, flag, 8, "start of time series");
        this.appendBitMaskInfo(info, flag, 16, "end of time series");
        this.appendBitMaskInfo(info, flag, 32, "clock locked");
        return "0x" + StringUtils.toHexString(flag, 2) + " (" + info + ")";
    }

    public int getDataQualityFlag() {
        return 0xFF & this.buffer.get(FixedHeader.DATA_QUALITY_FLAG.offset());
    }

    public String getDataQualityInfo() {
        int flag = this.getDataQualityFlag();
        if (flag == 0) {
            return "0x" + StringUtils.toHexString(flag, 2);
        }
        StringBuilder info = new StringBuilder();
        this.appendBitMaskInfo(info, flag, 1, "amplifier saturation detected");
        this.appendBitMaskInfo(info, flag, 2, "digitizer clipping detected");
        this.appendBitMaskInfo(info, flag, 4, "spikes detected");
        this.appendBitMaskInfo(info, flag, 8, "glitches detected");
        this.appendBitMaskInfo(info, flag, 16, "missing/padded data present");
        this.appendBitMaskInfo(info, flag, 32, "telemetry synchronization error");
        this.appendBitMaskInfo(info, flag, 64, "digital filter may be charging");
        this.appendBitMaskInfo(info, flag, 128, "questionable time tag");
        return "0x" + StringUtils.toHexString(flag, 2) + " (" + info + ")";
    }

    public int getTimingQualityValue() {
        if (this.hasBlockette(1001)) {
            byte quality = this.buffer.get(this.findBlockette(1001, this.buffer.capacity()) + Blockette1001.TIMING_QUALITY.offset());
            return quality & 0xFF;
        }
        return -1;
    }

    public Record setTimingQualityValue(byte quality) {
        if (quality < 0 || quality > 100) {
            throw new IllegalArgumentException("The timing quality is expected as a percentage from 0 to 100% maximum accuracy! (But was '" + quality + "'.)");
        }
        if (!this.hasBlockette(1001) && !this.injectBlockette1001()) {
            throw new UnsupportedException("Could not add a new blockette #1001 to the miniSEED record (#" + this.getSequenceString() + "' of station '" + this.getStationId() + "', channel '" + this.getChannelId() + "' at approx. " + this.getStartTime().toLogString() + ") and hence cannot set timing quality value ('" + quality + "').");
        }
        this.buffer.put(this.findBlockette(1001, this.buffer.capacity()) + Blockette1001.TIMING_QUALITY.offset(), quality);
        return this;
    }

    public String getTimingQualityInfo() {
        if (this.hasBlockette(1001)) {
            byte quality = this.buffer.get(this.findBlockette(1001, this.buffer.capacity()) + Blockette1001.TIMING_QUALITY.offset());
            return String.valueOf(quality & 0xFF);
        }
        return "N/A";
    }

    private static enum Blockette1001 {
        BLOCKETTE_ID(0, FieldType.UNSIGNED_INTEGER, 2),
        NEXT_BLOCKETTE(2, FieldType.UNSIGNED_INTEGER, 2),
        TIMING_QUALITY(4, FieldType.UNSIGNED_INTEGER, 1),
        TIME_MICROSECOND(5, FieldType.SIGNED_INTEGER, 1),
        PADDING(6, FieldType.ASCII_CHAR, 1),
        DATA_FRAMES(7, FieldType.UNSIGNED_INTEGER, 1);

        private final int offset;
        private final FieldType type;
        private final int length;

        private Blockette1001(int offset, FieldType type, int length) {
            this.offset = offset;
            this.type = type;
            this.length = length;
        }

        public int offset() {
            return this.offset;
        }

        public FieldType type() {
            return this.type;
        }

        public int length() {
            return this.length;
        }

        public static int totalSize() {
            return 8;
        }
    }

    private static enum Blockette1000 {
        BLOCKETTE_ID(0, FieldType.UNSIGNED_INTEGER, 2),
        NEXT_BLOCKETTE(2, FieldType.UNSIGNED_INTEGER, 2),
        DATA_FORMAT(4, FieldType.UNSIGNED_INTEGER, 1),
        BYTE_ORDER(5, FieldType.UNSIGNED_INTEGER, 1),
        RECORD_LENGTH(6, FieldType.UNSIGNED_INTEGER, 1),
        PADDING(7, FieldType.ASCII_CHAR, 1);

        private final int offset;
        private final FieldType type;
        private final int length;

        private Blockette1000(int offset, FieldType type, int length) {
            this.offset = offset;
            this.type = type;
            this.length = length;
        }

        public int offset() {
            return this.offset;
        }

        public FieldType type() {
            return this.type;
        }

        public int length() {
            return this.length;
        }

        public static int totalSize() {
            return 8;
        }
    }

    private static enum Blockette100 {
        BLOCKETTE_ID(0, FieldType.UNSIGNED_INTEGER, 2),
        NEXT_BLOCKETTE(2, FieldType.UNSIGNED_INTEGER, 2),
        SAMPLE_RATE(4, FieldType.FLOATING_POINT, 4),
        FLAG(8, FieldType.ASCII_CHAR, 1),
        PADDING(9, FieldType.ASCII_CHAR, 3);

        private final int offset;
        private final FieldType type;
        private final int length;

        private Blockette100(int offset, FieldType type, int length) {
            this.offset = offset;
            this.type = type;
            this.length = length;
        }

        public int offset() {
            return this.offset;
        }

        public FieldType type() {
            return this.type;
        }

        public int length() {
            return this.length;
        }

        public static int totalSize() {
            return 12;
        }
    }

    private static enum DataBlockette {
        BLOCKETTE_ID(0, FieldType.UNSIGNED_INTEGER, 2),
        NEXT_BLOCKETTE(2, FieldType.UNSIGNED_INTEGER, 2);

        private final int offset;
        private final FieldType type;
        private final int length;

        private DataBlockette(int offset, FieldType type, int length) {
            this.offset = offset;
            this.type = type;
            this.length = length;
        }

        public int offset() {
            return this.offset;
        }

        public FieldType type() {
            return this.type;
        }

        public int length() {
            return this.length;
        }
    }

    public static enum FixedHeader {
        SEQUENCE_NO(0, FieldType.ASCII_NUMBER, 6),
        DATA_INDICATOR(6, FieldType.ASCII_CHAR, 1),
        SPACE(7, FieldType.ASCII_CHAR, 1),
        STATION_ID(8, FieldType.ASCII_CHAR, 5),
        LOCATION_ID(13, FieldType.ASCII_CHAR, 2),
        CHANNEL_ID(15, FieldType.ASCII_CHAR, 3),
        NETWORK_ID(18, FieldType.ASCII_CHAR, 2),
        TIME_YEAR(20, FieldType.UNSIGNED_INTEGER, 2),
        TIME_DAYOFYEAR(22, FieldType.UNSIGNED_INTEGER, 2),
        TIME_HOUR(24, FieldType.UNSIGNED_INTEGER, 1),
        TIME_MINUTE(25, FieldType.UNSIGNED_INTEGER, 1),
        TIME_SECOND(26, FieldType.UNSIGNED_INTEGER, 1),
        PADDING(27, FieldType.ASCII_CHAR, 1),
        TIME_TICKS(28, FieldType.UNSIGNED_INTEGER, 2),
        SAMPLE_COUNT(30, FieldType.UNSIGNED_INTEGER, 2),
        SAMPLE_FACTOR(32, FieldType.SIGNED_INTEGER, 2),
        SAMPLE_MULTIPLIER(34, FieldType.SIGNED_INTEGER, 2),
        ACTIVITY_FLAG(36, FieldType.UNSIGNED_INTEGER, 1),
        IO_AND_CLOCK_FLAG(37, FieldType.UNSIGNED_INTEGER, 1),
        DATA_QUALITY_FLAG(38, FieldType.UNSIGNED_INTEGER, 1),
        BLOCKETTES(39, FieldType.UNSIGNED_INTEGER, 1),
        TICK_CORRECTION(40, FieldType.SIGNED_INTEGER, 4),
        BEGIN_DATA(44, FieldType.UNSIGNED_INTEGER, 2),
        BEGIN_BLOCKETTE(46, FieldType.UNSIGNED_INTEGER, 2);

        private final int offset;
        private final FieldType type;
        private final int length;

        private FixedHeader(int offset, FieldType type, int length) {
            this.offset = offset;
            this.type = type;
            this.length = length;
        }

        public int offset() {
            return this.offset;
        }

        public FieldType type() {
            return this.type;
        }

        public int length() {
            return this.length;
        }

        public static int totalSize() {
            return 48;
        }
    }

    private static enum FieldType {
        ASCII_CHAR,
        ASCII_NUMBER,
        SIGNED_INTEGER,
        UNSIGNED_INTEGER,
        FLOATING_POINT;

    }
}

